from fastapi import FastAPI, BackgroundTasks
from pydantic import BaseModel, Field, HttpUrl
import os, subprocess

OUTPUT_DIR = os.getenv("OUTPUT_DIR", "./output")
PUBLIC_BASE_URL = os.getenv("PUBLIC_BASE_URL", "https://YOUR-AI-SERVER-DOMAIN/files")
os.makedirs(OUTPUT_DIR, exist_ok=True)

app = FastAPI(title="SongAI Engine")

class JobIn(BaseModel):
    job_id: int = Field(..., ge=1)
    title: str
    lyrics: str
    style: str = "pop"
    tempo: int = 120
    callback_url: HttpUrl

def post_callback(url: str, payload: dict):
    import json
    data = json.dumps(payload, ensure_ascii=False)
    subprocess.run(
        ["curl", "-sS", "-X", "POST", "-H", "Content-Type: application/json", "-d", data, url],
        check=False,
        stdout=subprocess.DEVNULL,
        stderr=subprocess.DEVNULL,
    )

def generate_demo_song(job: JobIn):
    try:
        post_callback(str(job.callback_url), {"job_id": job.job_id, "status": "processing", "progress": 5})

        wav_path = os.path.join(OUTPUT_DIR, f"{job.job_id}.wav")
        mp3_path = os.path.join(OUTPUT_DIR, f"{job.job_id}.mp3")

        post_callback(str(job.callback_url), {"job_id": job.job_id, "status": "processing", "progress": 30})

        # DEMO sound (replace later with real AI models)
        subprocess.run(
            ["ffmpeg", "-y", "-f", "lavfi", "-i", "sine=frequency=220:duration=20", wav_path],
            check=True,
            stdout=subprocess.DEVNULL,
            stderr=subprocess.DEVNULL,
        )

        post_callback(str(job.callback_url), {"job_id": job.job_id, "status": "processing", "progress": 75})

        subprocess.run(
            ["ffmpeg", "-y", "-i", wav_path, "-codec:a", "libmp3lame", "-q:a", "4", mp3_path],
            check=True,
            stdout=subprocess.DEVNULL,
            stderr=subprocess.DEVNULL,
        )

        result_url = f"{PUBLIC_BASE_URL}/{job.job_id}.mp3"
        post_callback(str(job.callback_url), {"job_id": job.job_id, "status": "done", "progress": 100, "result_url": result_url})

    except Exception as e:
        post_callback(str(job.callback_url), {"job_id": job.job_id, "status": "failed", "progress": 0, "error_message": str(e)})

@app.post("/jobs")
def create_job(job: JobIn, bg: BackgroundTasks):
    post_callback(str(job.callback_url), {"job_id": job.job_id, "status": "queued", "progress": 0})
    bg.add_task(generate_demo_song, job)
    return {"ok": True, "job_id": job.job_id}

@app.get("/health")
def health():
    return {"ok": True}
